<?php
/**
 * Orders API for Toko Baju Junata
 * Handles order creation and retrieval
 */

// Error handler for development - converts PHP errors to JSON
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    $errorData = [
        'success' => false,
        'error' => 'PHP Error',
        'message' => $errstr,
        'file' => basename($errfile),
        'line' => $errline
    ];
    header('Content-Type: application/json');
    echo json_encode($errorData);
    exit;
});

register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'error' => 'Fatal Error',
            'message' => $error['message'],
            'file' => basename($error['file']),
            'line' => $error['line']
        ]);
    }
});

require_once 'config.php';

$requestData = getRequestData();
$action = isset($requestData['action']) ? $requestData['action'] : '';

switch ($action) {
    case 'create':
        createOrder($pdo, $requestData);
        break;
    
    case 'getByUser':
        getUserOrders($pdo, $requestData);
        break;
    
    case 'detail':
        getOrderDetail($pdo, $requestData);
        break;
    
    case 'updateStatus':
        updateOrderStatus($pdo, $requestData);
        break;
    
    default:
        sendResponse(false, 'Invalid action', null, 400);
        break;
}

/**
 * Create order from request data (items included in request)
 */
function createOrder($pdo, $data) {
    $required = ['user_id', 'nama_penerima', 'no_telp_penerima', 'alamat_pengiriman', 'payment_method', 'total_amount', 'items'];
    $missing = validateRequired($data, $required);
    
    if (!empty($missing)) {
        sendResponse(false, 'Missing fields: ' . implode(', ', $missing), null, 400);
    }
    
    // Validate items array
    if (!is_array($data['items']) || empty($data['items'])) {
        sendResponse(false, 'Items must be a non-empty array', null, 400);
    }
    
    $user_id = intval($data['user_id']);
    $nama_penerima = sanitizeInput($data['nama_penerima'] ?? '');
    $no_telp_penerima = sanitizeInput($data['no_telp_penerima'] ?? '');
    $alamat_pengiriman = sanitizeInput($data['alamat_pengiriman'] ?? '');
    $note = isset($data['note']) && $data['note'] ? sanitizeInput($data['note']) : '';
    $payment_method = sanitizeInput($data['payment_method'] ?? '');
    $total_amount = floatval($data['total_amount'] ?? 0);
    $items = $data['items'];
    
    try {
        // Start transaction
        $pdo->beginTransaction();
        
        // Insert order with total_amount
        $stmt = $pdo->prepare("
            INSERT INTO orders (user_id, total_amount, nama_penerima, no_telp_penerima, 
                              alamat_pengiriman, note, payment_method, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'Dikirim')
        ");
        
        $stmt->execute([
            $user_id,
            $total_amount,
            $nama_penerima,
            $no_telp_penerima,
            $alamat_pengiriman,
            $note,
            $payment_method
        ]);
        
        $order_id = $pdo->lastInsertId();
        
        // Insert order items
        $stmt = $pdo->prepare("
            INSERT INTO order_items (order_id, product_id, quantity, price, subtotal)
            VALUES (?, ?, ?, ?, ?)
        ");
        
        foreach ($items as $item) {
            if (!isset($item['product_id']) || !isset($item['quantity']) || !isset($item['harga'])) {
                $pdo->rollBack();
                sendResponse(false, 'Invalid item data', null, 400);
            }
            
            $product_id = intval($item['product_id']);
            $quantity = intval($item['quantity']);
            $price = floatval($item['harga']);
            $subtotal = $quantity * $price;
            
            $stmt->execute([$order_id, $product_id, $quantity, $price, $subtotal]);
            
            // Update product stock
            $updateStmt = $pdo->prepare("UPDATE products SET stok = stok - ? WHERE id = ?");
            $updateStmt->execute([$quantity, $product_id]);
        }
        
        // Clear user's cart
        $clearCart = $pdo->prepare("DELETE FROM carts WHERE user_id = ?");
        $clearCart->execute([$user_id]);
        
        // Commit transaction
        $pdo->commit();
        
        // Get created order with items count
        $stmt = $pdo->prepare("
            SELECT o.*, COUNT(oi.id) as total_items 
            FROM orders o 
            LEFT JOIN order_items oi ON o.id = oi.order_id 
            WHERE o.id = ? 
            GROUP BY o.id
        ");
        $stmt->execute([$order_id]);
        $order = $stmt->fetch();
        
        sendResponse(true, 'Order created successfully', $order, 201);
        
    } catch (PDOException $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        logError("Create order error: " . $e->getMessage());
        sendResponse(false, 'Failed to create order: ' . $e->getMessage(), null, 500);
    }
}

/**
 * Get all orders for a user
 */
function getUserOrders($pdo, $data) {
    if (!isset($data['user_id']) || empty($data['user_id'])) {
        sendResponse(false, 'User ID is required', null, 400);
    }
    
    $user_id = intval($data['user_id']);
    
    try {
        // Use stored procedure
        $stmt = $pdo->prepare("CALL sp_get_user_orders(?)");
        $stmt->execute([$user_id]);
        $orders = $stmt->fetchAll();
        
        sendResponse(true, 'Orders retrieved successfully', $orders, 200);
        
    } catch (PDOException $e) {
        logError("Get user orders error: " . $e->getMessage());
        sendResponse(false, 'Failed to retrieve orders', null, 500);
    }
}

/**
 * Get order detail with items
 */
function getOrderDetail($pdo, $data) {
    if (!isset($data['id']) || empty($data['id'])) {
        sendResponse(false, 'Order ID is required', null, 400);
    }
    
    $order_id = intval($data['id']);
    
    try {
        // Use stored procedure
        $stmt = $pdo->prepare("CALL sp_get_order_detail(?)");
        $stmt->execute([$order_id]);
        
        // Get order header
        $order = $stmt->fetch();
        
        if (!$order) {
            sendResponse(false, 'Order not found', null, 404);
        }
        
        // Get order items (second result set)
        $stmt->nextRowset();
        $items = $stmt->fetchAll();
        
        $result = [
            'order' => $order,
            'items' => $items
        ];
        
        sendResponse(true, 'Order detail retrieved successfully', $result, 200);
        
    } catch (PDOException $e) {
        logError("Get order detail error: " . $e->getMessage());
        sendResponse(false, 'Failed to retrieve order detail', null, 500);
    }
}

/**
 * Update order status
 */
function updateOrderStatus($pdo, $data) {
    $required = ['order_id', 'status'];
    $missing = validateRequired($data, $required);
    
    if (!empty($missing)) {
        sendResponse(false, 'Missing required fields: ' . implode(', ', $missing), null, 400);
    }
    
    $order_id = intval($data['order_id']);
    $status = sanitizeInput($data['status']);
    
    // Validate status
    $validStatuses = ['Pending', 'Dikemas', 'Dikirim', 'Selesai', 'Dibatalkan'];
    if (!in_array($status, $validStatuses)) {
        sendResponse(false, 'Invalid status value', null, 400);
    }
    
    try {
        $stmt = $pdo->prepare("UPDATE orders SET status = ? WHERE id = ?");
        $stmt->execute([$status, $order_id]);
        
        if ($stmt->rowCount() === 0) {
            sendResponse(false, 'Order not found', null, 404);
        }
        
        sendResponse(true, 'Order status updated successfully', null, 200);
        
    } catch (PDOException $e) {
        logError("Update order status error: " . $e->getMessage());
        sendResponse(false, 'Failed to update order status', null, 500);
    }
}
