<?php
/**
 * Cart API for Toko Baju Junata
 * Handles shopping cart operations
 */

require_once 'config.php';

$requestData = getRequestData();
$action = isset($requestData['action']) ? $requestData['action'] : '';

switch ($action) {
    case 'add':
        addToCart($pdo, $requestData);
        break;
    
    case 'get':
        getCartItems($pdo, $requestData);
        break;
    
    case 'update':
        updateCartItem($pdo, $requestData);
        break;
    
    case 'remove':
        removeFromCart($pdo, $requestData);
        break;
    
    case 'clear':
        clearCart($pdo, $requestData);
        break;
    
    default:
        sendResponse(false, 'Invalid action', null, 400);
        break;
}

/**
 * Add product to cart
 */
function addToCart($pdo, $data) {
    $required = ['user_id', 'product_id', 'quantity'];
    $missing = validateRequired($data, $required);
    
    if (!empty($missing)) {
        sendResponse(false, 'Missing required fields: ' . implode(', ', $missing), null, 400);
    }
    
    $user_id = intval($data['user_id']);
    $product_id = intval($data['product_id']);
    $quantity = intval($data['quantity']);
    
    if ($quantity < 1) {
        sendResponse(false, 'Quantity must be at least 1', null, 400);
    }
    
    try {
        // Check if product exists and has enough stock
        $stmt = $pdo->prepare("SELECT id, nama, stok FROM products WHERE id = ?");
        $stmt->execute([$product_id]);
        $product = $stmt->fetch();
        
        if (!$product) {
            sendResponse(false, 'Product not found', null, 404);
        }
        
        if ($product['stok'] < $quantity) {
            sendResponse(false, 'Insufficient stock. Available: ' . $product['stok'], null, 400);
        }
        
        // Check if item already in cart
        $stmt = $pdo->prepare("SELECT id, quantity FROM carts WHERE user_id = ? AND product_id = ?");
        $stmt->execute([$user_id, $product_id]);
        $existingCart = $stmt->fetch();
        
        if ($existingCart) {
            // Update quantity if already exists
            $newQuantity = $existingCart['quantity'] + $quantity;
            
            if ($newQuantity > $product['stok']) {
                sendResponse(false, 'Total quantity exceeds available stock', null, 400);
            }
            
            $stmt = $pdo->prepare("UPDATE carts SET quantity = ? WHERE id = ?");
            $stmt->execute([$newQuantity, $existingCart['id']]);
            
            sendResponse(true, 'Cart updated successfully', ['cart_id' => $existingCart['id']], 200);
        } else {
            // Insert new cart item
            $stmt = $pdo->prepare("INSERT INTO carts (user_id, product_id, quantity) VALUES (?, ?, ?)");
            $stmt->execute([$user_id, $product_id, $quantity]);
            
            $cartId = $pdo->lastInsertId();
            sendResponse(true, 'Added to cart successfully', ['cart_id' => $cartId], 201);
        }
        
    } catch (PDOException $e) {
        logError("Add to cart error: " . $e->getMessage());
        sendResponse(false, 'Failed to add to cart', null, 500);
    }
}

/**
 * Get cart items for a user
 */
function getCartItems($pdo, $data) {
    if (!isset($data['user_id']) || empty($data['user_id'])) {
        sendResponse(false, 'User ID is required', null, 400);
    }
    
    $user_id = intval($data['user_id']);
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM v_cart_details WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->execute([$user_id]);
        $cartItems = $stmt->fetchAll();
        
        // Calculate total
        $total = 0;
        foreach ($cartItems as $item) {
            $total += $item['subtotal'];
        }
        
        $result = [
            'items' => $cartItems,
            'total_items' => count($cartItems),
            'total_amount' => $total
        ];
        
        sendResponse(true, 'Cart items retrieved successfully', $result, 200);
        
    } catch (PDOException $e) {
        logError("Get cart items error: " . $e->getMessage());
        sendResponse(false, 'Failed to retrieve cart items', null, 500);
    }
}

/**
 * Update cart item quantity
 */
function updateCartItem($pdo, $data) {
    $required = ['cart_id', 'quantity'];
    $missing = validateRequired($data, $required);
    
    if (!empty($missing)) {
        sendResponse(false, 'Missing required fields: ' . implode(', ', $missing), null, 400);
    }
    
    $cart_id = intval($data['cart_id']);
    $quantity = intval($data['quantity']);
    
    if ($quantity < 1) {
        sendResponse(false, 'Quantity must be at least 1', null, 400);
    }
    
    try {
        // Get cart item and product stock
        $stmt = $pdo->prepare("
            SELECT c.id, c.product_id, p.stok 
            FROM carts c 
            INNER JOIN products p ON c.product_id = p.id 
            WHERE c.id = ?
        ");
        $stmt->execute([$cart_id]);
        $cart = $stmt->fetch();
        
        if (!$cart) {
            sendResponse(false, 'Cart item not found', null, 404);
        }
        
        if ($quantity > $cart['stok']) {
            sendResponse(false, 'Quantity exceeds available stock', null, 400);
        }
        
        // Update quantity
        $stmt = $pdo->prepare("UPDATE carts SET quantity = ? WHERE id = ?");
        $stmt->execute([$quantity, $cart_id]);
        
        sendResponse(true, 'Cart item updated successfully', null, 200);
        
    } catch (PDOException $e) {
        logError("Update cart item error: " . $e->getMessage());
        sendResponse(false, 'Failed to update cart item', null, 500);
    }
}

/**
 * Remove item from cart
 */
function removeFromCart($pdo, $data) {
    if (!isset($data['id']) || empty($data['id'])) {
        sendResponse(false, 'Cart ID is required', null, 400);
    }
    
    $cart_id = intval($data['id']);
    
    try {
        $stmt = $pdo->prepare("DELETE FROM carts WHERE id = ?");
        $stmt->execute([$cart_id]);
        
        if ($stmt->rowCount() === 0) {
            sendResponse(false, 'Cart item not found', null, 404);
        }
        
        sendResponse(true, 'Item removed from cart successfully', null, 200);
        
    } catch (PDOException $e) {
        logError("Remove from cart error: " . $e->getMessage());
        sendResponse(false, 'Failed to remove item from cart', null, 500);
    }
}

/**
 * Clear all cart items for a user
 */
function clearCart($pdo, $data) {
    if (!isset($data['user_id']) || empty($data['user_id'])) {
        sendResponse(false, 'User ID is required', null, 400);
    }
    
    $user_id = intval($data['user_id']);
    
    try {
        $stmt = $pdo->prepare("DELETE FROM carts WHERE user_id = ?");
        $stmt->execute([$user_id]);
        
        sendResponse(true, 'Cart cleared successfully', null, 200);
        
    } catch (PDOException $e) {
        logError("Clear cart error: " . $e->getMessage());
        sendResponse(false, 'Failed to clear cart', null, 500);
    }
}
