<?php
/**
 * Authentication API for Toko Baju Junata
 * Handles user registration and login
 */

require_once 'config.php';

$requestData = getRequestData();
$action = isset($requestData['action']) ? $requestData['action'] : '';

switch ($action) {
    case 'register':
        register($pdo, $requestData);
        break;
    
    case 'login':
        login($pdo, $requestData);
        break;
    
    default:
        sendResponse(false, 'Invalid action', null, 400);
        break;
}

/**
 * Register new user
 */
function register($pdo, $data) {
    // Validate required fields
    $required = ['nama', 'email', 'password', 'no_telp'];
    $missing = validateRequired($data, $required);
    
    if (!empty($missing)) {
        sendResponse(false, 'Missing required fields: ' . implode(', ', $missing), null, 400);
    }
    
    // Sanitize inputs
    $nama = sanitizeInput($data['nama']);
    $email = filter_var(sanitizeInput($data['email']), FILTER_VALIDATE_EMAIL);
    $password = $data['password'];
    $no_telp = sanitizeInput($data['no_telp']);
    $alamat = isset($data['alamat']) ? sanitizeInput($data['alamat']) : '';
    
    // Validate email format
    if (!$email) {
        sendResponse(false, 'Invalid email format', null, 400);
    }
    
    // Validate password length
    if (strlen($password) < 6) {
        sendResponse(false, 'Password must be at least 6 characters', null, 400);
    }
    
    try {
        // Check if email already exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);
        
        if ($stmt->fetch()) {
            sendResponse(false, 'Email already registered', null, 409);
        }
        
        // Hash password
        $hashedPassword = password_hash($password, PASSWORD_BCRYPT);
        
        // Insert new user
        $stmt = $pdo->prepare("
            INSERT INTO users (nama, email, password, no_telp, alamat) 
            VALUES (?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([$nama, $email, $hashedPassword, $no_telp, $alamat]);
        
        // Get the newly created user
        $userId = $pdo->lastInsertId();
        $stmt = $pdo->prepare("SELECT id, nama, email, no_telp, alamat, foto_profil, created_at FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        sendResponse(true, 'Registration successful', $user, 201);
        
    } catch (PDOException $e) {
        logError("Register error: " . $e->getMessage());
        sendResponse(false, 'Registration failed: ' . $e->getMessage(), null, 500);
    }
}

/**
 * Login user
 */
function login($pdo, $data) {
    // Validate required fields
    $required = ['email', 'password'];
    $missing = validateRequired($data, $required);
    
    if (!empty($missing)) {
        sendResponse(false, 'Missing required fields: ' . implode(', ', $missing), null, 400);
    }
    
    $email = filter_var(sanitizeInput($data['email']), FILTER_VALIDATE_EMAIL);
    $password = $data['password'];
    
    // Validate email format
    if (!$email) {
        sendResponse(false, 'Invalid email format', null, 400);
    }
    
    try {
        // Get user by email
        $stmt = $pdo->prepare("
            SELECT id, nama, email, password, no_telp, alamat, foto_profil, created_at 
            FROM users 
            WHERE email = ?
        ");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        if (!$user) {
            sendResponse(false, 'Invalid email or password', null, 401);
        }
        
        // Verify password
        if (!password_verify($password, $user['password'])) {
            sendResponse(false, 'Invalid email or password', null, 401);
        }
        
        // Remove password from response
        unset($user['password']);
        
        sendResponse(true, 'Login successful', $user, 200);
        
    } catch (PDOException $e) {
        logError("Login error: " . $e->getMessage());
        sendResponse(false, 'Login failed: ' . $e->getMessage(), null, 500);
    }
}
